--
-- TripComputerField
--
-- Author: Sławek Jaskulski
-- Copyright (C) Mod Next, All Rights Reserved.
--

-- name of the mod
local modName = g_currentModName

TripComputerField = {}

---Checks if all prerequisite specializations are present
function TripComputerField.prerequisitesPresent(specializations)
  return true
end

---Registers functions for this specialization
-- @param vehicleType table The vehicle type to register functions for
function TripComputerField.registerFunctions(vehicleType)
  SpecializationUtil.registerFunction(vehicleType, "getFieldInfo", TripComputerField.getFieldInfo)
end

---Registers event listeners for this specialization
-- @param vehicleType table The vehicle type to register listeners for
function TripComputerField.registerEventListeners(vehicleType)
  SpecializationUtil.registerEventListener(vehicleType, "onLoad", TripComputerField)
  SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", TripComputerField)
end

---Initializes the specialization when vehicle is loaded
-- @param savegame table Savegame data (unused)
function TripComputerField:onLoad(savegame)
  self.spec_tripComputerField = self[("spec_%s.tripComputerField"):format(modName)]
  local spec = self.spec_tripComputerField

  -- field state tracking
  spec.isOnField = false
  spec.isOnFieldSmoothed = false

  -- farmland information
  spec.farmlandId = 0
  spec.isOnFieldLastPos = { 0, 0 }

  -- distance-based update system (meters)
  spec.lastUpdateDistance = 0
  spec.updateDistance = 10
end

---Updates field detection state on each tick
function TripComputerField:onUpdateTick(dt, isActive, isActiveForInput, isSelected)
  local spec = self.spec_tripComputerField

  -- accumulate distance traveled
  spec.lastUpdateDistance = spec.lastUpdateDistance + (self.lastMovedDistance or 0)

  -- update field information periodically or when farmlandId is unknown
  if spec.lastUpdateDistance > spec.updateDistance or spec.farmlandId == 0 then
    spec.lastUpdateDistance = 0

    local x, _, z = getWorldTranslation(self.rootNode)
    spec.farmlandId = g_farmlandManager:getFarmlandIdAtWorldPosition(x, z)
    local isOnField = self:getIsOnField()

    -- handle field state changes
    if isOnField ~= spec.isOnField then
      if isOnField then
        spec.isOnFieldSmoothed = true
      else
        spec.isOnFieldLastPos = { x, z }
      end
    end

    -- apply smoothing when leaving field (prevent flickering)
    if spec.isOnFieldSmoothed ~= isOnField then
      local distance = MathUtil.vector2Length(x - spec.isOnFieldLastPos[1], z - spec.isOnFieldLastPos[2])
      if distance > 20 then
        spec.isOnFieldSmoothed = isOnField
      end
    end

    spec.isOnField = isOnField
  end
end

---Gets current field information
-- @return boolean isOnField Whether vehicle is currently on a field
-- @return number farmlandId The ID of the current farmland
-- @return boolean isOnFieldSmoothed Smoothed field state (prevents flickering)
function TripComputerField:getFieldInfo()
  local spec = self.spec_tripComputerField
  return spec.isOnField, spec.farmlandId, spec.isOnFieldSmoothed
end
