--
-- TripComputerUtil
--
-- Author: Sławek Jaskulski
-- Copyright (C) Mod Next, All Rights Reserved.
--

TripComputerUtil = {}

TripComputerUtil_mt = Class(TripComputerUtil)

-- dashboard value types available for XML configuration
TripComputerUtil.dashboardParams = {
  "mode",
  "drivenDistance",
  "drivenDistanceKm",
  "drivenDistanceMiles",
  "fuelUsage",
  "totalFuel",
  "avgFuelConsumption",
  "avgFuelConsumptionMiles",
  "avgFuelConsumptionHour",
  "avgFuelConsumptionHourMiles",
  "range",
  "rangeMiles",
  "rangeHour",
  "operatingTime",
  "resetState",
}

---Calculates fuel usage with unit conversion
-- Converts liters to gallons when using imperial units
-- @param number fuelUsed fuel consumed in liters
-- @return number fuel usage in liters or gallons
function TripComputerUtil:getFuelUsage(fuelUsed)
  if fuelUsed == nil then
    return 0
  end

  if not g_gameSettings:getValue("useMiles") then
    return fuelUsed
  end

  -- convert liters to gallons (1 gallon = 3.785 liters)
  return fuelUsed / 3.785
end

---Calculates instantaneous fuel consumption per distance
-- Converts L/h consumption to L/100km (metric) or MPG (imperial) based on current speed
-- @param number fuelUsagePerHour current fuel consumption in liters per hour
-- @param number currentSpeedKmh current speed in km/h
-- @param boolean useMiles optional, use imperial units (default: from game settings)
-- @return number fuel consumption per distance (L/100km or MPG)
function TripComputerUtil:getFuelUsagePerKm(fuelUsagePerHour, currentSpeedKmh, useMiles)
  if fuelUsagePerHour == nil or currentSpeedKmh == nil then
    return 0
  end

  if currentSpeedKmh <= 0 then
    return 0
  end

  if useMiles == nil then
    useMiles = g_gameSettings:getValue("useMiles")
  end

  if useMiles then
    -- convert to MPG (miles per gallon)
    local currentSpeedMph = currentSpeedKmh * 0.621371
    local fuelUsageGph = fuelUsagePerHour * 0.264172

    if fuelUsageGph <= 0 then
      return 0
    end

    return currentSpeedMph / fuelUsageGph
  end

  -- convert L/h to L/100km: (L/h) / (km/h) * 100
  return (fuelUsagePerHour / currentSpeedKmh) * 100
end

---Converts distance from meters to kilometers or miles
-- @param number distance distance in meters
-- @param boolean useMiles optional, use imperial units (default: from game settings)
-- @return number distance in kilometers or miles
function TripComputerUtil:getDistance(distance, useMiles)
  if distance == nil then
    return 0
  end

  if useMiles == nil then
    useMiles = g_gameSettings:getValue("useMiles")
  end

  -- convert meters to kilometers
  distance = distance / 1000

  if useMiles then
    -- convert kilometers to miles
    distance = distance * 0.621371
  end

  return distance
end

---Calculates average fuel consumption over total distance
-- Returns L/100km (metric) or MPG (imperial)
-- @param number fuelUsed total fuel consumed in liters
-- @param number distance total distance in meters
-- @param boolean useMiles optional, use imperial units (default: from game settings)
-- @return number average fuel consumption (L/100km or MPG)
function TripComputerUtil:getAvgFuelConsumption(fuelUsed, distance, useMiles)
  if fuelUsed == nil or distance == nil then
    return 0
  end

  if distance <= 0 or fuelUsed <= 0 then
    return 0
  end

  if useMiles == nil then
    useMiles = g_gameSettings:getValue("useMiles")
  end

  -- convert meters to kilometers
  local distanceKm = distance * 0.001

  if useMiles then
    -- calculate MPG (miles per gallon)
    local distanceMiles = distanceKm * 0.621371
    local fuelGallons = fuelUsed * 0.264172
    return distanceMiles / fuelGallons
  end

  -- calculate L/100km
  return (fuelUsed / distanceKm) * 100
end

---Calculates average fuel consumption per hour
-- Returns L/h (metric) or gal/h (imperial)
-- @param number fuelUsed total fuel consumed in liters
-- @param number operatingTime total operating time in milliseconds
-- @param boolean useMiles optional, use imperial units (default: from game settings)
-- @return number fuel consumption per hour (L/h or gal/h)
function TripComputerUtil:getFuelConsumptionHour(fuelUsed, operatingTime, useMiles)
  if fuelUsed == nil or operatingTime == nil then
    return 0
  end

  if operatingTime <= 0 or fuelUsed <= 0 then
    return 0
  end

  if useMiles == nil then
    useMiles = g_gameSettings:getValue("useMiles")
  end

  -- convert milliseconds to hours
  local operatingHours = operatingTime / (1000 * 60 * 60)

  if useMiles then
    -- calculate gallons per hour
    local fuelGallons = fuelUsed * 0.264172
    return fuelGallons / operatingHours
  end

  -- calculate liters per hour
  return fuelUsed / operatingHours
end

---Calculates remaining range based on fuel level and consumption
-- Returns range in km/mi (distance-based) or hours (time-based)
-- @param number fuelLevel current fuel level in liters
-- @param number avgConsumption average fuel consumption (L/100km or MPG)
-- @param boolean useMiles optional, use imperial units (default: from game settings)
-- @return number remaining range in kilometers or miles
function TripComputerUtil:getRange(fuelLevel, avgConsumption, useMiles)
  if fuelLevel == nil or avgConsumption == nil then
    return 0
  end

  if fuelLevel <= 0 or avgConsumption <= 0 then
    return 0
  end

  if useMiles == nil then
    useMiles = g_gameSettings:getValue("useMiles")
  end

  -- calculate range in kilometers
  local rangeKm = (fuelLevel * 100) / avgConsumption

  if useMiles then
    -- convert to miles
    return rangeKm * 0.621371
  end

  return rangeKm
end

---Calculates remaining operating time based on fuel level
-- Returns remaining time in hours based on current consumption rate
-- @param number fuelLevel current fuel level in liters
-- @param number consumptionPerHour current consumption rate in liters per hour
-- @return number remaining operating time in hours
function TripComputerUtil:getRangeHour(fuelLevel, consumptionPerHour)
  if fuelLevel == nil or consumptionPerHour == nil then
    return 0
  end

  if fuelLevel <= 0 or consumptionPerHour <= 0 then
    return 0
  end

  -- calculate hours remaining
  return fuelLevel / consumptionPerHour
end

---Converts operating time to hours and minutes format
-- Converts milliseconds to hours.minutes format (e.g., 1.3 = 1h 30min)
-- @param number operatingTime operating time in milliseconds
-- @return number hours hours component
-- @return number minutes minutes component (0-5 representing 0, 10, 20, 30, 40, 50 minutes)
function TripComputerUtil:getOperatingTime(operatingTime)
  if operatingTime == nil then
    return 0, 0
  end

  if operatingTime <= 0 then
    return 0, 0
  end

  -- convert milliseconds to minutes
  local minutes = operatingTime / (1000 * 60)
  local hours = math.floor(minutes / 60)
  -- convert to 0-5 range (10-minute intervals)
  local remainingMinutes = math.floor((minutes - hours * 60) / 6)

  return hours, remainingMinutes
end

---Retrieves fuel level and type for the vehicle
-- Checks for diesel, electric charge, and methane fuel types
-- @param table vehicle vehicle instance to check
-- @return number fuelLevel current fuel level in liters (or kWh for electric)
-- @return number fuelType FillType constant (DIESEL, ELECTRICCHARGE, or METHANE)
function TripComputerUtil:getFuelInfo(vehicle)
  if vehicle == nil then
    return 0, nil
  end

  -- check for supported fuel types in priority order
  local fuelTypes = {
    FillType.DIESEL,
    FillType.ELECTRICCHARGE,
    FillType.METHANE,
  }

  for _, fuelType in ipairs(fuelTypes) do
    local fillUnitIndex = vehicle:getConsumerFillUnitIndex(fuelType)

    if fillUnitIndex ~= nil then
      local level = vehicle:getFillUnitFillLevel(fillUnitIndex)

      return level, fuelType
    end
  end

  return 0, nil
end

---Determines display unit for fuel usage based on fuel type
-- Returns localized unit text (e.g., "L/h", "gal/h", "kW/h", "kg/h")
-- @param table vehicle vehicle instance to check fuel type
-- @param boolean useLongName optional, use long unit names (default: short names)
-- @return string unit text for fuel usage display
function TripComputerUtil:getFuelUsageUnit(vehicle, useLongName)
  if vehicle == nil then
    return ""
  end

  local _, fuelType = TripComputerUtil:getFuelInfo(vehicle)
  local postfix = useLongName and "" or "Short"

  -- map fuel types to their corresponding unit keys
  local unitMap = {
    [FillType.DIESEL] = {
      miles = "unit_tripComputerGallonsPerHour",
      metric = "unit_tripComputerLiter",
    },
    [FillType.METHANE] = {
      miles = "unit_tripComputerCubicFoot",
      metric = "unit_tripComputerKilogram",
    },
    [FillType.ELECTRICCHARGE] = "unit_tripComputerKilowatt",
  }

  local fuelUnit = unitMap[fuelType]

  if fuelUnit ~= nil then
    local useMiles = g_gameSettings:getValue("useMiles")

    -- get localized fuel unit text
    local fuelUnitText
    if type(fuelUnit) == "table" then
      local unitKey
      if useMiles then
        unitKey = fuelUnit.miles
      else
        unitKey = fuelUnit.metric
      end
      fuelUnitText = g_i18n:getText(unitKey .. postfix)
    else
      fuelUnitText = g_i18n:getText(fuelUnit .. postfix)
    end

    -- for imperial diesel, gallons/hour is already complete
    if useMiles and fuelType == FillType.DIESEL then
      return fuelUnitText
    end

    -- append "/h" for other units
    local hourText = g_i18n:getText("unit_tripComputerHour" .. postfix)

    return fuelUnitText .. "/" .. hourText
  end

  return ""
end

---Determines display unit for average fuel consumption
-- Returns localized unit text (e.g., "L/100km", "MPG", "kWh/100km")
-- @param table vehicle vehicle instance to check fuel type
-- @param boolean useLongName optional, use long unit names (default: short names)
-- @return string unit text for average consumption display
function TripComputerUtil:getAvgFuelConsumptionUnit(vehicle, useLongName)
  if vehicle == nil then
    return ""
  end

  local _, fuelType = TripComputerUtil:getFuelInfo(vehicle)
  local postfix = useLongName and "" or "Short"

  -- map fuel types to their corresponding unit keys
  local unitMap = {
    [FillType.DIESEL] = {
      miles = "unit_tripComputerMilesPerGallon",
      metric = "unit_tripComputerLiter",
    },
    [FillType.METHANE] = {
      miles = "unit_tripComputerCubicFoot",
      metric = "unit_tripComputerKilogram",
    },
    [FillType.ELECTRICCHARGE] = "unit_tripComputerKilowattHour",
  }

  local fuelUnit = unitMap[fuelType]
  if fuelUnit ~= nil then
    local useMiles = g_gameSettings:getValue("useMiles")

    -- for imperial diesel, MPG is already complete
    if useMiles and fuelType == FillType.DIESEL then
      return g_i18n:getText(fuelUnit.miles .. postfix)
    end

    -- get localized fuel unit text
    local fuelUnitText
    if type(fuelUnit) == "table" then
      local unitKey
      if useMiles then
        unitKey = fuelUnit.miles
      else
        unitKey = fuelUnit.metric
      end
      fuelUnitText = g_i18n:getText(unitKey .. postfix)
    else
      fuelUnitText = g_i18n:getText(fuelUnit .. postfix)
    end

    -- append "/100km" or "/100mi"
    local measuringUnit = g_i18n:getMeasuringUnit()
    local distanceUnitText = "100" .. measuringUnit

    return fuelUnitText .. "/" .. distanceUnitText
  end

  return ""
end

---Checks if vehicle has a fuel system
-- @param table vehicle vehicle instance to check
-- @return boolean true if vehicle has fuel system
function TripComputerUtil:hasFuelSystem(vehicle)
  local _, fuelType = TripComputerUtil:getFuelInfo(vehicle)
  return fuelType ~= nil
end
