--
-- TripComputerBuffer
--
-- Author: Sławek Jaskulski
-- Copyright (C) Mod Next, All Rights Reserved.
--

TripComputerBuffer = {}

local TripComputerBuffer_mt = Class(TripComputerBuffer)

---Creates a new instance of TripComputerBuffer
function TripComputerBuffer.new(bufferSize)
  local self = setmetatable({}, TripComputerBuffer_mt)

  -- initialize buffer properties
  self.size = math.floor(bufferSize)
  self.index = 0
  self.sum = 0
  self.sumSquares = 0
  self.values = {}

  return self
end

---Adds a new value to the circular buffer
---Only operates on server side to prevent desync
-- @param value number The value to add to the buffer
function TripComputerBuffer:add(value)
  if g_server == nil then
    return
  end

  -- circular buffer: wrap around when reaching buffer size
  local nextIndex = (self.index % self.size) + 1
  local oldValue = self.values[nextIndex] or 0

  self.values[nextIndex] = value
  -- update running sums incrementally for efficient average calculation
  self.sum = self.sum + value - oldValue
  self.sumSquares = self.sumSquares + (value * value) - (oldValue * oldValue)
  self.index = nextIndex
end

---Gets the most recently added value from the buffer
-- @return number The last added value, or 0 if buffer is empty
function TripComputerBuffer:get()
  if self.index == 0 then
    return 0 -- buffer is empty
  end
  return self.values[self.index] or 0
end

---Gets the filtered average value of the buffer using statistical outlier removal
---Uses 2-standard-deviation rule to filter outliers
-- @return number The filtered average value, or 0 if buffer is empty
function TripComputerBuffer:getAverage()
  local sampleCount = math.min(#self.values, self.size)

  if sampleCount == 0 then
    return 0
  end

  -- calculate statistics for outlier filtering
  local mean = self.sum / sampleCount
  local variance = math.max(0, (self.sumSquares / sampleCount) - (mean * mean))
  local stdDev = math.sqrt(variance)
  local threshold = 2 * stdDev -- 2-standard-deviation rule for outliers

  -- filter outliers and calculate average
  local filteredSum, filteredCount = 0, 0
  for i = 1, sampleCount do
    local value = self.values[i]

    if value ~= nil and math.abs(value - mean) <= threshold then
      filteredSum = filteredSum + value
      filteredCount = filteredCount + 1
    end
  end

  return filteredCount > 0 and (filteredSum / filteredCount) or mean
end

---Resets the buffer to its initial empty state
function TripComputerBuffer:reset()
  self.index = 0
  self.sum = 0
  self.sumSquares = 0
  self.values = {}
end
