-- MoreDesignConfigs for FS25
-- Extends design configurations from 16 to 49 and designColor from 16 to 49
--
-- Allows using:
--   design17 to design49 configurations
--   designColor17 to designColor49 configurations
--
-- Note: FS25 already includes design2-16 and designColor2-16 natively
--
-- Usage: Add this as a specialization to your vehicle mod's modDesc.xml:
--   <specializations>
--       <specialization name="moreDesignConfigs" className="MoreDesignConfigs" filename="MoreDesignConfigs.lua"/>
--   </specializations>
--
--   <vehicleTypes>
--       <type name="myVehicle" parent="baseDrivable" ...>
--           <specialization name="moreDesignConfigs"/>
--       </type>
--   </vehicleTypes>

MoreDesignConfigs = {}

-- FS25 already provides design2-16, so we start at 17
MoreDesignConfigs.DESIGN_START = 17
MoreDesignConfigs.DESIGN_END = 49

-- FS25 already provides designColor2-16, so we start at 17
MoreDesignConfigs.DESIGN_COLOR_START = 17
MoreDesignConfigs.DESIGN_COLOR_END = 49

-- Track if we've already registered (prevent duplicate registration)
MoreDesignConfigs.initialized = false

function MoreDesignConfigs.prerequisitesPresent(specializations)
    return true
end

function MoreDesignConfigs.initSpecialization()
    -- Only initialize once, even if multiple vehicles use this specialization
    if MoreDesignConfigs.initialized then
        return
    end
    MoreDesignConfigs.initialized = true

    -- Register additional design configurations (design17 to design49)
    for i = MoreDesignConfigs.DESIGN_START, MoreDesignConfigs.DESIGN_END do
        local name = string.format("design%d", i)

        -- Check if already registered (in case another mod already added it)
        if g_vehicleConfigurationManager:getConfigurationIndexByName(name) == nil then
            g_vehicleConfigurationManager:addConfigurationType(
                name,                                    -- configuration name
                g_i18n:getText("configuration_design"),  -- display title
                nil,                                     -- xml key (nil = auto-generate)
                VehicleConfigurationItem                 -- item class
            )
        end
    end

    -- Register additional designColor configurations (designColor17 to designColor49)
    for i = MoreDesignConfigs.DESIGN_COLOR_START, MoreDesignConfigs.DESIGN_COLOR_END do
        local name = string.format("designColor%d", i)

        -- Check if already registered
        if g_vehicleConfigurationManager:getConfigurationIndexByName(name) == nil then
            g_vehicleConfigurationManager:addConfigurationType(
                name,                                         -- configuration name
                g_i18n:getText("configuration_designColor"),  -- display title
                nil,                                          -- xml key (nil = auto-generate)
                VehicleConfigurationItemColor                 -- item class for colors
            )
        end
    end

    -- Register XML paths for the schema
    local schema = Vehicle.xmlSchema
    if schema ~= nil then
        for i = MoreDesignConfigs.DESIGN_START, MoreDesignConfigs.DESIGN_END do
            local configName = string.format("design%d", i)
            local configurationsKey = string.format("vehicle.%sConfigurations", configName)
            local configurationKey = string.format("%s.%sConfiguration(?)", configurationsKey, configName)

            VehicleConfigurationItem.registerXMLPaths(schema, configurationsKey, configurationKey)
            ObjectChangeUtil.registerObjectChangeXMLPaths(schema, configurationKey)
        end

        for i = MoreDesignConfigs.DESIGN_COLOR_START, MoreDesignConfigs.DESIGN_COLOR_END do
            local configName = string.format("designColor%d", i)
            local configurationsKey = string.format("vehicle.%sConfigurations", configName)
            local configurationKey = string.format("%s.%sConfiguration(?)", configurationsKey, configName)

            VehicleConfigurationItemColor.registerXMLPaths(schema, configurationsKey, configurationKey)
            ObjectChangeUtil.registerObjectChangeXMLPaths(schema, configurationKey)
        end
    end
end

function MoreDesignConfigs.registerFunctions(vehicleType)
    -- No custom functions needed
end

function MoreDesignConfigs.registerEventListeners(vehicleType)
    -- No custom event listeners needed - VehicleConfigurationItem and
    -- VehicleConfigurationItemColor handle everything automatically via
    -- their onLoad/onPostLoad callbacks
end
